<?php
/*
  This plugin will localize the following things:
  1. Configuration that can is loaded by BConfig::LoadFrom() - that is, *.conf and other
     files (mottos.txt, etc.). Note that it won't load/install plugins.
     * localized config will be merged with default one so you can only leave files and
       options (in info.conf and others) there that are different from your default config.
  2. Post texts - depending on a setting you can have translated posts either under the
     same directory but with a suffix added to post file name or under a different directory
     retaining original names.
     * you don't have to localize every post - if no translation a found original text
       will be shown.
     * posts are formatted with UWiki config of the user that has added that post (i.e. the
       admin). So unless you change to desired language before adding a post 'uwiki config path'
       setting in target language's config file won't affect it (but it will work for things
       like user-added comments which are formatted according to their interface language).
     * {{DateWritten}} and other fields are required in both original and translated posts
       but some of them (e.g. DateWritten) are ignored for the latter psots; however, you
       can still use others (such as Category) and they will work. Experiment to find out :)

  Post localization only changes texts (which is probably what you want it to do anyway) -
  so view counts, comments and such will all be in one place. However, juggling with config
  files (for example, setting 'comments path' to different values) you can separate some
  areas of your blog.

  You don't have to reindex old posts after installing this plugin.

  In terms of speed the blog engine works as such that if user uses default language to access
  the blog this plugin does essentially nothing.
  Technical side is that this plugin redirects certain actions, in fact creating for them
  two (or more) blogs at one place.
*/

class BLocali {
  const CookieName = 'blog-language';
  const CookieExpiresIn = 2592000;    // seconds; default is 30 days.

  // the first language will be considered the default.
  static $languages = array('ru', 'en');
  // relative to _blog; {LANG} will be replaced with visitor's language code.
  static $configPath = 'config-{LANG}';

  // post directory will be read from localized config.
  static $translatePosts = true;
  // Can be empty; if it isn't then post file name is: dir/dir/.../file[suffixLANG][postExt].
  // Don't set this to empty and 'posts path' to the same location as that of default
  // language as there will be no way to distinguish localized posts from original ones.
  static $postSuffix = '.';   // so '.' makes post.en.wiki, '-' makes post-en.wiki, etc.
  // will automatically add tag named as lang (e.g. "en") to localized posts.
  static $addLangTag = true;
  // will set interface language for localized posts according to it for spiders like
  // Google (for example, posts with "ru" tag will have Russian interface (provided it's
  // specified in $languages above) even if Accept-Language that robot sent allows English).
  static $defaultLangForRobots = true;
  // Specifies if visitors with a language different from default should be redirected
  // when they're coming on the site main page ($siteHome). Can be an array of form
  // "lang" => (false|true|"url") where "url" (a string) is their new location;
  // "$" symbol there is replaced with visitor's language code (e.g. "ru"). If it's not
  // a string but true visitors go to the post list filtered by tag "LANG", e.g. url
  // "=LANG" (e.g. "=ru"); if it's false visitor with that particular language isn't redirected.
  // Additionally, $redirect can be not an array but just true, false or a string - in this case
  // redirection happens (true) or not happens (false) for all BUT visitors with default
  // language - they are never redirected. A string is an URL where "$" is replaced with the language.
  // All URLs can be either relative (expanded to site home URL) or absolute (untouched).
  // For example: $redirect = array('ru' => 'http://google.ru', 'en' => '=English');
  static $redirect = true;

  static $detectedLang = '';

  // returns null for default language.
  static function UserLang() {
    $lang = &self::$detectedLang;

    if ($lang === '') {
      $places = array(&$_GET['language'], &$_POST['language'], &$_GET['lang'], &$_POST['lang']);
      while (!$lang and count($places)) { $lang = array_shift($places); }

      // only set robot lang if it wasn't explicitly passed (cookies and Accept-Language don't count):
      if (!$lang and self::$defaultLangForRobots and IsRobotAgent()) {
        BEvent::Hook('post viewed', array(__CLASS__, 'SetRobotsLangFor'));
      }

      $lang or $lang = @$_COOKIE[self::CookieName];
      $lang or $lang = self::DetectUserLang();

      self::Exists($lang) or $lang = null;

      setcookie(self::CookieName, $lang, time() + self::CookieExpiresIn, parse_url(BConfig::$siteHome, PHP_URL_PATH));
      $lang === self::$languages[0] and $lang = null;
    }

    return $lang;
  }

    static function Exists($lang) { return in_array($lang, self::$languages); }

    static function DetectUserLang() {
      $accepts = @$_SERVER['HTTP_ACCEPT_LANGUAGE'];

      if ($accepts) {
        $languages = array();

        $accepts = explode(',', $accepts);
        foreach ($accepts as $piece) {
          @list($lang, $quality) = explode(';', $piece);

          if (@$quality and substr($quality, 0, 2) == 'q=') {
            $quality = floatval(trim( substr($quality, 2) ));
          } else {
            $quality = 1.0;
          }

          $languages[ round($quality * 1000) ] = trim($lang);  // *1000 because ksort() discards float values.
        }

        krsort($languages);

        foreach ($languages as $lang) {
          $lang = substr(trim($lang), 0, 2);  // = ISO-2
          if (self::Exists($lang)) { return $lang; }
        }
      }
    }

    static function SetRobotsLangFor($post) {
      $tags = BPosts::Get($post, 'tags');

      foreach (self::$languages as $lang) {
        if (in_array($lang, $tags)) {
          self::$detectedLang = $lang;
          self::LoadLangConfig();
          break;
        }
      }
    }

  function LoadLangConfig() {
    if (($config = self::ConfigPath()) and is_dir($config)) {
      BConfig::LoadFrom($config);
    }
  }

    static function ConfigPath() {
      if (self::UserLang()) {
        $path = self::$configPath;
        return str_replace('{LANG}', self::UserLang(), ToAbsolutePath($path));
      }
    }

      static function PostSuffix() {
        return self::$postSuffix.self::UserLang() .BConfig::$postFileExt;
      }

  static function OnStart() {
    self::TryRedirecting();

    if (self::UserLang()) {
      self::LoadLangConfig();

      if (self::$translatePosts and self::$postSuffix) {
        BEvent::HookFirst('index: file load', array(__CLASS__, 'FileIndexLoad'));
      }
    }

    if (self::$translatePosts) {
      BEvent::HookFirst('post added', array(__CLASS__, 'PostAdded'));
      BEvent::HookFirst('post updated', array(__CLASS__, 'PostAdded'));
    }
  }

    static function TryRedirecting() {
      $currentURL = 'http://'.$_SERVER['HTTP_HOST'].$_SERVER['REQUEST_URI'];
      if (rtrim($currentURL, '\\/') === rtrim(BConfig::$siteHome, '\\/')) {
        $lang = self::UserLang();

        if (is_array(self::$redirect) or $lang) {
          $lang or $lang = self::$languages[0];
          $redir = is_array(self::$redirect) ? @self::$redirect[$lang] : self::$redirect;

          if ($redir) {
            $url = $redir === true ? "=$lang" : str_replace('$', $lang, $redir);
            RedirectTo( ToAbsoluteURL($url) );
          }
        }
      }
    }

  static function OriginalPostOf($post, &$lang) {
    $suffix = preg_quote(self::$postSuffix, '~');
    $ext = preg_quote(BConfig::$postFileExt, '~');
    $langs = join('|', array_slice(self::$languages, 1));

    if (preg_match("~^(.+)$suffix($langs)($ext)$~", $post, $match)) {
      $lang = $match[2];
      return $match[1].$match[3];
    } else {
      $lang = null;
    }
  }

    static function PostAdded($post, &$info) {
      if (self::$addLangTag and self::OriginalPostOf($post, $lang)) {
        if (!in_array($lang, $info['tags'])) {
          $info['tags'][] = $lang;
          sort($info['tags']);
        }
      }

      // Post can be (1) original, (2) localized. If no suffxi is used (localized posts
      // reside in different folders) (1) & (2) are differentiated as: UserLang() ? (2) : (1).
      if (self::$postSuffix) {
        $post = self::OriginalPostOf($post, $lang);
        $isOrig = $post === null;
      } else {
        $isOrig = self::UserLang() === null;
        $lang = self::UserLang();
      }

      if (!$isOrig) {
        $interlangPost = $post.".$lang";
        $isUpdate = (bool) BIndex::SelectFrom('PostDocs', $interlangPost);
        BIndex::AddTo('PostDocs', $interlangPost, $info);

        foreach ($info['tags'] as $tag) {
          if ($isUpdate) {
            BIndex::RemoveFrom('tags', $tag, $post);
            BIndex::RemoveFrom('TagCounts', $tag);
          }

          BIndex::AddTo('tags', $tag, $post);
          BIndex::AddTo('TagCounts', $tag);
        }

        return true;
      }
    }

    static function FileIndexLoad($index, &$data) {
      // index = post-docs/.../post.wiki[.jp]
      if ($index[0] === 'p') {
        $userSuffix = '.'.self::UserLang();
        if (substr($index, -3) !== $userSuffix) {
          $data = BaseFileIndex::Load($index.$userSuffix);
          return $data ? true : null;
        }
      }
    }
}

  BEvent::Hook('on start', array('BLocali', 'OnStart'));

/* one-shot */
  $this->name = 'localization';
  $this->Caption('Мультиязычность', 'ru');
